import { useState, useEffect, useRef, useCallback } from "react";

// ─── Maze Generation (Recursive Backtracker) ────────────────────────────────
function generateMaze(cols, rows) {
  const grid = Array.from({ length: rows }, (_, r) =>
    Array.from({ length: cols }, (_, c) => ({
      r,
      c,
      walls: { N: true, S: true, E: true, W: true },
      visited: false,
    })),
  );
  const cell = (r, c) =>
    r >= 0 && r < rows && c >= 0 && c < cols ? grid[r][c] : null;
  const stack = [];
  const start = grid[0][0];
  start.visited = true;
  stack.push(start);
  const dirs = [
    { d: "N", dr: -1, dc: 0, opp: "S" },
    { d: "S", dr: 1, dc: 0, opp: "N" },
    { d: "E", dr: 0, dc: 1, opp: "W" },
    { d: "W", dr: 0, dc: -1, opp: "E" },
  ];
  while (stack.length) {
    const cur = stack[stack.length - 1];
    const neighbors = dirs
      .map(({ d, dr, dc, opp }) => ({
        neighbor: cell(cur.r + dr, cur.c + dc),
        d,
        opp,
      }))
      .filter(({ neighbor }) => neighbor && !neighbor.visited);
    if (!neighbors.length) {
      stack.pop();
      continue;
    }
    const { neighbor, d, opp } =
      neighbors[Math.floor(Math.random() * neighbors.length)];
    cur.walls[d] = false;
    neighbor.walls[opp] = false;
    neighbor.visited = true;
    stack.push(neighbor);
  }
  return grid;
}

// ─── Constants ───────────────────────────────────────────────────────────────
const COLS = 15,
  ROWS = 12,
  CELL = 44;
const W = COLS * CELL,
  H = ROWS * CELL;
const FOG_RADIUS = 3;

// ─── Ink brushstroke colours ─────────────────────────────────────────────────
const PALETTE = {
  bg: "#f5f0e8",
  paper: "#ede8dc",
  ink: "#1a1410",
  inkLight: "#3d342a",
  orb: "#c8703a",
  orbGlow: "#e8a060",
  exit: "#4a7c59",
  fog: "#f5f0e8",
  trail: "#c8703a",
};

function lerp(a, b, t) {
  return a + (b - a) * t;
}

export default function MindLabyrinth() {
  const canvasRef = useRef(null);
  const stateRef = useRef(null);
  const keysRef = useRef({});
  const animRef = useRef(null);
  const [phase, setPhase] = useState("start"); // start | playing | win
  const [level, setLevel] = useState(1);
  const [moves, setMoves] = useState(0);
  const [time, setTime] = useState(0);
  const timerRef = useRef(null);

  const initLevel = useCallback((lvl) => {
    const cols = Math.min(COLS, 9 + lvl);
    const rows = Math.min(ROWS, 7 + Math.floor(lvl / 2));
    const maze = generateMaze(cols, rows);
    const trail = Array.from({ length: rows }, () => Array(cols).fill(0));
    trail[0][0] = 1;
    stateRef.current = {
      maze,
      cols,
      rows,
      player: {
        r: 0,
        c: 0,
        x: CELL / 2,
        y: CELL / 2,
        targetX: CELL / 2,
        targetY: CELL / 2,
      },
      exit: { r: rows - 1, c: cols - 1 },
      trail,
      revealed: Array.from({ length: rows }, () => Array(cols).fill(false)),
      moveQueue: [],
      moving: false,
      moveTimer: 0,
      particles: [],
      orbPulse: 0,
      won: false,
    };
    // Reveal initial fog
    revealFog(stateRef.current, 0, 0, cols, rows);
    setMoves(0);
    setTime(0);
    clearInterval(timerRef.current);
    timerRef.current = setInterval(() => setTime((t) => t + 1), 1000);
  }, []);

  function revealFog(s, r, c, cols, rows) {
    for (let dr = -FOG_RADIUS; dr <= FOG_RADIUS; dr++) {
      for (let dc = -FOG_RADIUS; dc <= FOG_RADIUS; dc++) {
        const nr = r + dr,
          nc = c + dc;
        if (nr >= 0 && nr < rows && nc >= 0 && nc < cols) {
          if (Math.abs(dr) + Math.abs(dc) <= FOG_RADIUS + 1)
            s.revealed[nr][nc] = true;
        }
      }
    }
  }

  // Input
  useEffect(() => {
    const down = (e) => {
      keysRef.current[e.code] = true;
      if (
        [
          "ArrowUp",
          "ArrowDown",
          "ArrowLeft",
          "ArrowRight",
          "KeyW",
          "KeyA",
          "KeyS",
          "KeyD",
        ].includes(e.code)
      )
        e.preventDefault();
    };
    const up = (e) => {
      keysRef.current[e.code] = false;
    };
    window.addEventListener("keydown", down);
    window.addEventListener("keyup", up);
    return () => {
      window.removeEventListener("keydown", down);
      window.removeEventListener("keyup", up);
    };
  }, []);

  // Game loop
  useEffect(() => {
    if (phase !== "playing") return;
    const canvas = canvasRef.current;
    const ctx = canvas.getContext("2d");

    let lastKey = null;
    let keyRepeat = 0;

    const loop = () => {
      const s = stateRef.current;
      if (!s) {
        animRef.current = requestAnimationFrame(loop);
        return;
      }
      const k = keysRef.current;
      s.orbPulse += 0.04;

      // Key input → movement
      const dirMap = [
        { codes: ["ArrowUp", "KeyW"], dr: -1, dc: 0, wall: "N" },
        { codes: ["ArrowDown", "KeyS"], dr: 1, dc: 0, wall: "S" },
        { codes: ["ArrowLeft", "KeyA"], dr: 0, dc: -1, wall: "W" },
        { codes: ["ArrowRight", "KeyD"], dr: 0, dc: 1, wall: "E" },
      ];

      if (!s.moving) {
        for (const { codes, dr, dc, wall } of dirMap) {
          const pressed = codes.some((c) => k[c]);
          if (pressed) {
            const cur = s.maze[s.player.r][s.player.c];
            const nr = s.player.r + dr,
              nc = s.player.c + dc;
            if (
              !cur.walls[wall] &&
              nr >= 0 &&
              nr < s.rows &&
              nc >= 0 &&
              nc < s.cols
            ) {
              s.moving = true;
              s.moveTimer = 0;
              s.player.fromX = s.player.x;
              s.player.fromY = s.player.y;
              s.player.r = nr;
              s.player.c = nc;
              s.player.targetX = nc * CELL + CELL / 2;
              s.player.targetY = nr * CELL + CELL / 2;
              revealFog(s, nr, nc, s.cols, s.rows);
              s.trail[nr][nc] = 1;
              // Particles
              for (let i = 0; i < 5; i++) {
                s.particles.push({
                  x: s.player.x,
                  y: s.player.y,
                  vx: (Math.random() - 0.5) * 1.5,
                  vy: (Math.random() - 0.5) * 1.5,
                  life: 1,
                  size: Math.random() * 3 + 1,
                });
              }
              setMoves((m) => m + 1);
              if (nr === s.exit.r && nc === s.exit.c) {
                s.won = true;
                setTimeout(() => {
                  setPhase("win");
                  clearInterval(timerRef.current);
                }, 700);
              }
            }
            break;
          }
        }
      }

      // Smooth movement
      if (s.moving) {
        s.moveTimer += 0.18;
        const t = Math.min(1, s.moveTimer);
        const ease = 1 - Math.pow(1 - t, 3);
        s.player.x = lerp(s.player.fromX, s.player.targetX, ease);
        s.player.y = lerp(s.player.fromY, s.player.targetY, ease);
        if (t >= 1) {
          s.moving = false;
        }
      }

      // Particles
      s.particles.forEach((p) => {
        p.x += p.vx;
        p.y += p.vy;
        p.life -= 0.06;
      });
      s.particles = s.particles.filter((p) => p.life > 0);

      // ── Draw ──────────────────────────────────────────────────────────────
      ctx.clearRect(0, 0, W, H);

      // Paper texture background
      ctx.fillStyle = PALETTE.paper;
      ctx.fillRect(0, 0, W, H);

      // Subtle grid grain
      ctx.save();
      for (let i = 0; i < 300; i++) {
        const gx = Math.random() * W,
          gy = Math.random() * H;
        ctx.fillStyle = `rgba(0,0,0,${Math.random() * 0.015})`;
        ctx.fillRect(gx, gy, 1, 1);
      }
      ctx.restore();

      // Trail cells
      for (let r = 0; r < s.rows; r++) {
        for (let c = 0; c < s.cols; c++) {
          if (!s.revealed[r][c]) continue;
          if (s.trail[r][c]) {
            ctx.fillStyle = "rgba(200,112,58,0.07)";
            ctx.fillRect(c * CELL + 2, r * CELL + 2, CELL - 4, CELL - 4);
          }
        }
      }

      // Exit marker
      if (s.revealed[s.exit.r][s.exit.c]) {
        const ex = s.exit.c * CELL + CELL / 2,
          ey = s.exit.r * CELL + CELL / 2;
        const pulse = 0.5 + 0.5 * Math.sin(s.orbPulse * 1.5);
        ctx.save();
        ctx.shadowColor = PALETTE.exit;
        ctx.shadowBlur = 12 + pulse * 8;
        ctx.fillStyle = PALETTE.exit;
        // Draw torii-like symbol
        ctx.beginPath();
        ctx.arc(ex, ey, 10, 0, Math.PI * 2);
        ctx.fill();
        ctx.fillStyle = "#fff";
        ctx.beginPath();
        ctx.arc(ex, ey, 5, 0, Math.PI * 2);
        ctx.fill();
        ctx.restore();

        ctx.fillStyle = PALETTE.exit + "aa";
        ctx.font = "bold 8px serif";
        ctx.textAlign = "center";
        ctx.fillText("EXIT", ex, ey + 18);
      }

      // Maze walls (ink style)
      ctx.lineCap = "round";
      for (let r = 0; r < s.rows; r++) {
        for (let c = 0; c < s.cols; c++) {
          const cell = s.maze[r][c];
          const revealed = s.revealed[r][c];
          const x = c * CELL,
            y = r * CELL;

          // Fog overlay
          if (!revealed) {
            ctx.fillStyle = PALETTE.fog;
            ctx.fillRect(x, y, CELL, CELL);
            continue;
          }

          const alpha = 1;
          const inkColor = PALETTE.ink;

          // Draw walls with ink brush variation
          const wallWidth = 2.5;
          ctx.strokeStyle = inkColor;
          ctx.lineWidth = wallWidth;

          if (cell.walls.N) {
            ctx.beginPath();
            ctx.moveTo(x + 1, y + 1);
            ctx.lineTo(x + CELL - 1, y + 1);
            ctx.globalAlpha = alpha;
            ctx.stroke();
            ctx.globalAlpha = 1;
          }
          if (cell.walls.S) {
            ctx.beginPath();
            ctx.moveTo(x + 1, y + CELL - 1);
            ctx.lineTo(x + CELL - 1, y + CELL - 1);
            ctx.globalAlpha = alpha;
            ctx.stroke();
            ctx.globalAlpha = 1;
          }
          if (cell.walls.W) {
            ctx.beginPath();
            ctx.moveTo(x + 1, y + 1);
            ctx.lineTo(x + 1, y + CELL - 1);
            ctx.globalAlpha = alpha;
            ctx.stroke();
            ctx.globalAlpha = 1;
          }
          if (cell.walls.E) {
            ctx.beginPath();
            ctx.moveTo(x + CELL - 1, y + 1);
            ctx.lineTo(x + CELL - 1, y + CELL - 1);
            ctx.globalAlpha = alpha;
            ctx.stroke();
            ctx.globalAlpha = 1;
          }
        }
      }

      // Particles
      s.particles.forEach((p) => {
        ctx.beginPath();
        ctx.arc(p.x, p.y, p.size, 0, Math.PI * 2);
        ctx.fillStyle = `rgba(200,112,58,${p.life * 0.6})`;
        ctx.fill();
      });

      // Player orb
      const pulse = 0.5 + 0.5 * Math.sin(s.orbPulse * 2);
      ctx.save();
      ctx.shadowColor = PALETTE.orbGlow;
      ctx.shadowBlur = 14 + pulse * 10;
      // Outer glow ring
      ctx.beginPath();
      ctx.arc(s.player.x, s.player.y, 11 + pulse * 2, 0, Math.PI * 2);
      ctx.fillStyle = `rgba(232,160,96,0.15)`;
      ctx.fill();
      // Main orb
      const grad = ctx.createRadialGradient(
        s.player.x - 3,
        s.player.y - 3,
        1,
        s.player.x,
        s.player.y,
        10,
      );
      grad.addColorStop(0, "#f0c080");
      grad.addColorStop(0.5, PALETTE.orb);
      grad.addColorStop(1, "#8a4020");
      ctx.beginPath();
      ctx.arc(s.player.x, s.player.y, 9, 0, Math.PI * 2);
      ctx.fillStyle = grad;
      ctx.fill();
      ctx.restore();

      // Fog gradient edges on revealed area
      for (let r = 0; r < s.rows; r++) {
        for (let c = 0; c < s.cols; c++) {
          if (!s.revealed[r][c]) continue;
          // Check if any neighbor is unrevealed → draw soft fog fade
          const neighbors = [
            [r - 1, c],
            [r + 1, c],
            [r, c - 1],
            [r, c + 1],
            [r - 1, c - 1],
            [r - 1, c + 1],
            [r + 1, c - 1],
            [r + 1, c + 1],
          ];
          const hasUnrevealedNeighbor = neighbors.some(
            ([nr, nc]) =>
              nr >= 0 &&
              nr < s.rows &&
              nc >= 0 &&
              nc < s.cols &&
              !s.revealed[nr][nc],
          );
          if (hasUnrevealedNeighbor) {
            const x = c * CELL,
              y = r * CELL;
            const fg = ctx.createRadialGradient(
              x + CELL / 2,
              y + CELL / 2,
              CELL * 0.2,
              x + CELL / 2,
              y + CELL / 2,
              CELL * 0.9,
            );
            fg.addColorStop(0, "rgba(245,240,232,0)");
            fg.addColorStop(1, "rgba(245,240,232,0.5)");
            ctx.fillStyle = fg;
            ctx.fillRect(x, y, CELL, CELL);
          }
        }
      }

      animRef.current = requestAnimationFrame(loop);
    };

    animRef.current = requestAnimationFrame(loop);
    return () => {
      cancelAnimationFrame(animRef.current);
      clearInterval(timerRef.current);
    };
  }, [phase]);

  const handleStart = () => {
    initLevel(1);
    setLevel(1);
    setPhase("playing");
  };

  const handleNext = () => {
    const next = level + 1;
    setLevel(next);
    initLevel(next);
    setPhase("playing");
  };

  const fmtTime = (s) =>
    `${String(Math.floor(s / 60)).padStart(2, "0")}:${String(s % 60).padStart(2, "0")}`;

  return (
    <div
      className="min-h-screen flex flex-col items-center justify-center"
      style={{ background: "#e8e0d0", fontFamily: "'Georgia', serif" }}
    >
      {/* Title */}
      <div className="mb-4 text-center select-none">
        <h1
          className="text-5xl font-bold mb-1"
          style={{
            color: "#1a1410",
            letterSpacing: "0.08em",
            textShadow: "2px 2px 0 #c8b89044",
            fontFamily: "'Palatino Linotype', 'Book Antiqua', serif",
          }}
        >
          Mind Labyrinth
        </h1>
        <p
          style={{
            color: "#6b5c4a",
            fontSize: "0.85rem",
            letterSpacing: "0.2em",
          }}
        >
          A ZEN MAZE JOURNEY
        </p>
      </div>

      {/* HUD */}
      {phase === "playing" && (
        <div
          className="flex justify-between items-center mb-3 px-2"
          style={{ width: W, maxWidth: "98vw" }}
        >
          <div style={{ color: "#6b5c4a" }}>
            <span
              style={{
                fontSize: "0.65rem",
                letterSpacing: "0.2em",
                textTransform: "uppercase",
              }}
            >
              Level
            </span>
            <div
              style={{
                fontSize: "1.6rem",
                fontWeight: "bold",
                color: "#1a1410",
                lineHeight: 1,
              }}
            >
              {level}
            </div>
          </div>
          <div className="text-center" style={{ color: "#6b5c4a" }}>
            <span
              style={{
                fontSize: "0.65rem",
                letterSpacing: "0.2em",
                textTransform: "uppercase",
              }}
            >
              Moves
            </span>
            <div
              style={{
                fontSize: "1.6rem",
                fontWeight: "bold",
                color: "#c8703a",
                lineHeight: 1,
              }}
            >
              {moves}
            </div>
          </div>
          <div className="text-right" style={{ color: "#6b5c4a" }}>
            <span
              style={{
                fontSize: "0.65rem",
                letterSpacing: "0.2em",
                textTransform: "uppercase",
              }}
            >
              Time
            </span>
            <div
              style={{
                fontSize: "1.6rem",
                fontWeight: "bold",
                color: "#1a1410",
                lineHeight: 1,
              }}
            >
              {fmtTime(time)}
            </div>
          </div>
        </div>
      )}

      {/* Canvas */}
      <div
        className="relative"
        style={{
          width: W,
          maxWidth: "98vw",
          boxShadow: "0 8px 40px rgba(0,0,0,0.18), 0 2px 8px rgba(0,0,0,0.10)",
          border: "1px solid #c8b890",
        }}
      >
        <canvas
          ref={canvasRef}
          width={W}
          height={H}
          style={{
            display: "block",
            maxWidth: "98vw",
            background: PALETTE.paper,
          }}
        />

        {/* Start Screen */}
        {phase === "start" && (
          <div
            className="absolute inset-0 flex flex-col items-center justify-center"
            style={{ background: "rgba(237,232,220,0.96)" }}
          >
            <div style={{ fontSize: "4rem", marginBottom: "0.2rem" }}>🌿</div>
            <h2
              style={{
                fontFamily: "'Palatino Linotype',serif",
                fontSize: "2rem",
                color: "#1a1410",
                marginBottom: "0.4rem",
              }}
            >
              Find the Path
            </h2>
            <p
              style={{
                color: "#6b5c4a",
                fontSize: "0.85rem",
                letterSpacing: "0.15em",
                marginBottom: "2rem",
                textAlign: "center",
                maxWidth: 320,
              }}
            >
              Navigate the labyrinth.
              <br />
              The fog reveals only what you've explored.
            </p>
            <button
              onClick={handleStart}
              style={{
                padding: "0.7rem 2.4rem",
                background: "#1a1410",
                color: "#f5f0e8",
                border: "none",
                fontSize: "0.9rem",
                letterSpacing: "0.2em",
                textTransform: "uppercase",
                cursor: "pointer",
                fontFamily: "'Georgia', serif",
                boxShadow: "0 4px 16px rgba(0,0,0,0.2)",
                transition: "all 0.2s",
              }}
              onMouseEnter={(e) => {
                e.target.style.background = "#c8703a";
              }}
              onMouseLeave={(e) => {
                e.target.style.background = "#1a1410";
              }}
            >
              Begin Journey
            </button>
            <p
              style={{
                color: "#aaa",
                fontSize: "0.7rem",
                marginTop: "1.4rem",
                letterSpacing: "0.1em",
              }}
            >
              WASD / Arrow Keys to move
            </p>
          </div>
        )}

        {/* Win Screen */}
        {phase === "win" && (
          <div
            className="absolute inset-0 flex flex-col items-center justify-center"
            style={{ background: "rgba(237,232,220,0.96)" }}
          >
            <div style={{ fontSize: "3.5rem", marginBottom: "0.3rem" }}>🪷</div>
            <h2
              style={{
                fontFamily: "'Palatino Linotype',serif",
                fontSize: "2rem",
                color: "#4a7c59",
                marginBottom: "0.3rem",
              }}
            >
              Path Complete
            </h2>
            <div
              style={{
                color: "#6b5c4a",
                marginBottom: "0.2rem",
                letterSpacing: "0.1em",
              }}
            >
              Level <strong>{level}</strong> · <strong>{moves}</strong> moves ·{" "}
              <strong>{fmtTime(time)}</strong>
            </div>
            <p
              style={{
                color: "#999",
                fontSize: "0.8rem",
                marginBottom: "1.8rem",
              }}
            >
              The mind grows still. The path grows deeper.
            </p>
            <button
              onClick={handleNext}
              style={{
                padding: "0.7rem 2.4rem",
                background: "#4a7c59",
                color: "#f5f0e8",
                border: "none",
                fontSize: "0.9rem",
                letterSpacing: "0.2em",
                textTransform: "uppercase",
                cursor: "pointer",
                fontFamily: "'Georgia', serif",
                boxShadow: "0 4px 16px rgba(0,0,0,0.15)",
                marginBottom: "0.7rem",
              }}
              onMouseEnter={(e) => {
                e.target.style.background = "#3a6249";
              }}
              onMouseLeave={(e) => {
                e.target.style.background = "#4a7c59";
              }}
            >
              Next Level →
            </button>
            <button
              onClick={handleStart}
              style={{
                padding: "0.4rem 1.4rem",
                background: "transparent",
                color: "#6b5c4a",
                border: "1px solid #c8b890",
                fontSize: "0.75rem",
                letterSpacing: "0.15em",
                textTransform: "uppercase",
                cursor: "pointer",
                fontFamily: "'Georgia',serif",
              }}
            >
              Restart
            </button>
          </div>
        )}
      </div>

      {/* Hint */}
      {phase === "playing" && (
        <p
          style={{
            color: "#aaa",
            fontSize: "0.7rem",
            marginTop: "0.8rem",
            letterSpacing: "0.12em",
            textTransform: "uppercase",
          }}
        >
          WASD / Arrow Keys · Find the green exit
        </p>
      )}
    </div>
  );
}
